use strict;
use File::Spec;

my $buildno = '0.1.2014.02.06';

print(STDERR <<"_END");
clclassseq $buildno
=======================================================================

Official web site of this script is
http://www.fifthdimension.jp/products/claident/ .
To know script details, see above URL.

Copyright (C) 2011-2014  Akifumi S. Tanabe

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

_END

# display usage if command line options were not specified
unless (@ARGV) {
	&helpMessage();
}

my $devnull = File::Spec->devnull();
my $numthreads = 1;
my $assamsoption;
my $outputfolder = $ARGV[-1];
my @inputfiles;
if (-e $outputfolder) {
	&errorMessage(__LINE__, "\"$outputfolder\" already exists.");
}

# get command line options
{
	my $assamsmode = 0;
	my %inputfiles;
	for (my $i = 0; $i < scalar(@ARGV) - 1; $i ++) {
		if ($ARGV[$i] eq 'end') {
			$assamsmode = 0;
		}
		elsif ($assamsmode) {
			$assamsoption .= " $ARGV[$i]";
		}
		elsif ($ARGV[$i] eq 'assams') {
			$assamsmode = 1;
		}
		elsif ($ARGV[$i] =~ /^-+(?:n|n(?:um)?threads?)=(\d+)$/i) {
			$numthreads = $1;
		}
		else {
			my @temp = glob($ARGV[$i]);
			if (scalar(@temp) > 0) {
				foreach (@temp) {
					if (!exists($inputfiles{$_})) {
						$inputfiles{$_} = 1;
						push(@inputfiles, $_);
					}
					else {
						&errorMessage(__LINE__, "\"$_\" is doubly specified.");
					}
				}
			}
			else {
				&errorMessage(__LINE__, "Input file does not exist.");
			}
		}
	}
}

if ($assamsoption =~ /-+(?:n|n(?:um)?threads?)=\d+/i) {
	&errorMessage(__LINE__, "The option for assams is invalid.");
}
if ($assamsoption !~ /-+min(?:imum)?n(?:um)?seqs?=\d+/i) {
	$assamsoption .= ' --minnseq=1000';
}
if ($assamsoption !~ /-+(?:min(?:imum)?ident(?:ity|ities)?|m)=\d+/i) {
	$assamsoption .= ' --minident=0.97';
}
if ($assamsoption !~ /-+min(?:imum)?(?:overlap|ovl)(?:length|len)=\d+/i) {
	$assamsoption .= ' --minovllen=100';
}
if ($assamsoption !~ /-+max(?:imum)?n(?:um)?hits?=\d+/i) {
	$assamsoption .= ' --maxnhits=0';
}
if ($assamsoption !~ /-+(?:strand|s)=(?:forward|plus|single|both|double)/i) {
	$assamsoption .= ' --strand=plus';
}
if ($assamsoption !~ /-+assemble(?:mode)?=(?:exact|ungapped|ungap|gapped|gap|e\+u\+g|e\+u|e\+g|u\+g|all|e|u|g|a)/i) {
	$assamsoption .= ' --assemblemode=u+g';
}
if ($assamsoption !~ /-+merge(?:mode)?=(rough|lazy|draft|rapid|fast|middle|moderate|normal|accurate|slow|strict)/i) {
	$assamsoption .= ' --mergemode=normal';
}
if ($assamsoption !~ /-+(?:(?:aln|align|alignment)wiggle|a)=\d+/i) {
	$assamsoption .= ' --alnwiggle=11';
}

if (!@inputfiles) {
	&errorMessage(__LINE__, "No input file was specified.");
}

print(STDERR "Command line options for assams for assembly:$assamsoption\n");
print(STDERR "\n");

my $bankreport;
{
	my $pathto;
	if ($ENV{'ASSAMSHOME'}) {
		$pathto = $ENV{'ASSAMSHOME'};
	}
	else {
		my $temp;
		if (-e '.assams') {
			$temp = '.assams';
		}
		elsif (-e $ENV{'HOME'} . '/.assams') {
			$temp = $ENV{'HOME'} . '/.assams';
		}
		elsif (-e '/etc/assams/.assams') {
			$temp = '/etc/assams/.assams';
		}
		if ($temp) {
			my $filehandle;
			unless (open($filehandle, "< $temp")) {
				&errorMessage(__LINE__, "Cannot read \"$temp\".");
			}
			while (<$filehandle>) {
				if (/^\s*ASSAMSHOME\s*=\s*(\S[^\r\n]*)/) {
					$pathto = $1;
					$pathto =~ s/\s+$//;
					last;
				}
			}
			close($filehandle);
		}
	}
	if ($pathto) {
		$pathto =~ s/^"(.+)"$/$1/;
		$pathto =~ s/\/$//;
		$pathto .= '/bin';
		if (!-e $pathto) {
			&errorMessage(__LINE__, "Cannot find \"$pathto\".");
		}
		$bankreport = "\"$pathto/bank-report\"";
	}
	else {
		$bankreport = 'bank-report';
	}
}

# make output directory
print(STDERR "Making output folder...\n");
if (!mkdir($outputfolder)) {
	&errorMessage(__LINE__, 'Cannot make output directory.');
}
print(STDERR "done.\n\n");

# run first assams in parallel
print(STDERR "Running assembly by assams at each file...\n");
{
	my @newinput;
	foreach my $inputfile (@inputfiles) {
		print(STDERR "Processing $inputfile...\n");
		my $filename = $inputfile;
		$filename =~ s/^.+(?:\/|\\)//;
		$filename =~ s/\.[^\.]+$//;
		push(@newinput, $filename);
		if (system("assams$assamsoption --numthreads=$numthreads $inputfile $outputfolder/$filename.bnk 2> $devnull 1> $devnull")) {
			&errorMessage(__LINE__, "Cannot run \"assams$assamsoption --numthreads=$numthreads $inputfile $outputfolder/$filename.bnk\".");
		}
	}
	@inputfiles = @newinput;
}
print(STDERR "done.\n\n");

# change working directory
unless (chdir($outputfolder)) {
	&errorMessage(__LINE__, 'Cannot change working directory.');
}

# postprocessing
print(STDERR "Deleting temporary files and generating AFG files...\n");
{
	my $child = 0;
	$| = 1;
	$? = 0;
	foreach my $inputfile (@inputfiles) {
		if (my $pid = fork()) {
			$child ++;
			if ($child >= ($numthreads + 1)) {
				if (wait == -1) {
					$child = 0;
				} else {
					$child --;
				}
			}
			if ($?) {
				&errorMessage(__LINE__, "The processes did not finished correctly.");
			}
			next;
		}
		else {
			# make afg
			if (system("$bankreport -p -b $inputfile.bnk 2> $devnull | gzip -9 -c > $inputfile.afg.gz 2> $devnull")) {
				&errorMessage(__LINE__, "Cannot run $bankreport -p|gzip for \"$inputfile.bnk\".");
			}
			# delete bank
			while (glob("$inputfile.bnk/*")) {
				unlink;
			}
			rmdir("$inputfile.bnk");
			exit;
		}
	}
	# join
	while (wait != -1) {
		if ($?) {
			&errorMessage(__LINE__, "The processes did not finished correctly.");
		}
	}
}
print(STDERR "done.\n\n");

chdir('..');

sub errorMessage {
	my $lineno = shift(@_);
	my $message = shift(@_);
	print(STDERR "ERROR!: line $lineno\n$message\n");
	print(STDERR "If you want to read help message, run this script without options.\n");
	exit(1);
}

sub helpMessage {
	print(STDERR <<"_END");
Usage
=====
clclassseq options inputfiles outputfolder

Command line options
====================
assams options end
  Specify commandline options for assams.
(default: --minnseq=1000 --minident=0.97 --minovllen=100 --maxnhits=0
 --strand=plus --assemblemode=u+g --mergemode=normal --alnwiggle=11)

-n, --numthreads=INTEGER
  Specify the number of processes. (default: 1)

Acceptable input file formats
=============================
FASTA (+.qual)
_END
	exit;
}
